using System;
using System.Data;
using gov.va.med.vbecs.DAL;
using gov.va.med.vbecs.ExceptionManagement;
using gov.va.med.vbecs.Common;
using VBECSLOCK = gov.va.med.vbecs.Common.VbecsTables.VbecsLock;

namespace gov.va.med.vbecs.BOL
{
	/// <summary>
	/// Summary description for LockManager.
	/// </summary>
	public class LockManager
	{
		/// <summary>
		/// Default lock timeout in minutes.
		/// </summary>
		public const int DefaultLockInactivityTimeoutMin = 5;

        //Property Variables
        private Common.LockFunctions _lockFunction = LockFunctions.NA;
        private int _lockedFormId = 0;
        private int _baseLockedFormId = int.MinValue;
	    private int _lockInactivityTimeoutMin = DefaultLockInactivityTimeoutMin;        //CR3311
        private System.Collections.ArrayList _recordGuidArray = new System.Collections.ArrayList();
        private static System.Collections.ArrayList _expiredBaseLockedFormIds = null;

        //Forms that are loaded during a locking-chain - used to close forms in VBECSBaseForm_Paint()
        private static System.Collections.Hashtable _lockedFormIdsTable = new System.Collections.Hashtable();
        private bool _isLockInitiator = false;
        private bool _lockingConflict = false;

        //LOCKING2.0-END

		#region Variables

		private System.Collections.ArrayList _locks;

		#endregion

        //Locking420
        #region Static Properties for Lock Timeout Management 


        /// <summary>
        /// LOCKING420- Stores a list of Base IDs whos locks have expired
        /// </summary>
        public static System.Collections.ArrayList ExpiredBaseLockedFormIds
        {
            get
            {
                if (_expiredBaseLockedFormIds == null)
                {
                    _expiredBaseLockedFormIds = new System.Collections.ArrayList();
                }
                return (_expiredBaseLockedFormIds);
            }
        }

        /// <summary>
        /// LOCKING420 - Add a form to a Hashtable. The Hashtable representst a list of forms that will need to close if the Base form lock expires
        /// </summary>
        /// <param name="baseLockedFormId"></param>
        /// <param name="nestedLockedFormId"></param>
        public static void AddToLockAlertList(int baseLockedFormId, int nestedLockedFormId)
        {
            if (_lockedFormIdsTable.Contains(nestedLockedFormId))
            {
                int tmpBaseLockedFormId = (int)_lockedFormIdsTable[nestedLockedFormId];
                if (tmpBaseLockedFormId != baseLockedFormId)
                    System.Diagnostics.Debug.WriteLine("Bug-AddToLockAlertList(" + baseLockedFormId.ToString() + "," + nestedLockedFormId.ToString() + ")");
            }
            else
            {
                _lockedFormIdsTable.Add(nestedLockedFormId, baseLockedFormId);
            }
        }

        /// <summary>
        /// LOCKING420 - Remove the Form guid from the Hasttable. This should only occur during Form Close events
        /// </summary>
        /// <param name="lockedFormId"></param>
        public static void RemoveFromLockAlertList(int lockedFormId)
        {
            if (_lockedFormIdsTable.Contains(lockedFormId))
                _lockedFormIdsTable.Remove(lockedFormId);
        }

        /// <summary>
        /// LOCKING420- Checks to see if there are any child forms left
        /// </summary>
        /// <param name="BaseLockedFormId"></param>
        /// <returns></returns>
        public static bool AllLockAlertFormsAreClosed(int BaseLockedFormId)
        {
            return(!_lockedFormIdsTable.ContainsValue(BaseLockedFormId));
        }

        /// <summary>
        /// LOCKING420 - Static member to identify if a BaseFormID's lock has timedout - set in the LockExpired() event
        /// </summary>
        /// <param name="BaseLockedFormId"></param>
        /// <returns></returns>
        public static bool HasBaseLockTimedout(int BaseLockedFormId)
        {
            return (ExpiredBaseLockedFormIds.Contains(BaseLockedFormId));
        }

        #endregion

        /// <summary>
        /// LOCKING420 - Used to determine if the form has locks, whether it started the lock or not. DOES NOT INDICATE A TIMEOUT.
        /// </summary>
        /// <param name="lockedFormId"></param>
        /// <returns></returns>
        public bool IsLockCarrier(int lockedFormId)
        {
            return (this._isLockInitiator || _lockedFormIdsTable.Contains(lockedFormId));
        }

        ///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/3/2005</CreationDate>
		/// <summary>
		/// Blank Constructor - Initializes our Locks Collection. 
		/// </summary>
        /// <param name="baseLockedFormId"></param>
        /// <param name="lockedFormId"></param>
        /// <param name="lockInactivityTimeoutMin"></param>
        public LockManager(int lockedFormId, int baseLockedFormId, int lockInactivityTimeoutMin)
		{
			this._locks = new System.Collections.ArrayList();
            this._lockedFormId = lockedFormId;
            this._baseLockedFormId = baseLockedFormId;
            this._lockInactivityTimeoutMin = lockInactivityTimeoutMin;      //CR3311
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/3/2005</CreationDate>
		/// <summary>
		/// Returns if the Record is Already Locked by the UC
		/// If it wasn't, any existing locks will get picked up when the lock attempt is
		/// performed.  This is a way to try and save DB calls
		/// </summary>
		/// <param name="recordGuid"></param>
		/// <param name="foundLock">Returns</param>
		/// <returns></returns>
		public bool IsRecordAlreadyLockedInUC(System.Guid recordGuid, out BOL.Lock foundLock)
		{
			bool hasRecord = false;
			BOL.Lock tmpLock = null;

			for (int x=0; x < this._locks.Count; x++)
			{
				tmpLock = (BOL.Lock)this._locks[x];

				if (tmpLock.LockedRecordGuid == recordGuid)
				{
					hasRecord = true;
					break;
				}
			    tmpLock = null;     //Needs to be reset back to null since nothing was found
			}

			foundLock = tmpLock;
			return(hasRecord);
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/3/2005</CreationDate>
		/// <summary>
		/// Returns the UserId of the person who has the lock
		/// </summary>
		/// <param name="recordGuid"></param>
		/// <param name="setLockConflict">Always set this to FALSE, except for ExceptionHandle in Insert</param>
		/// <returns></returns>
		public static BOL.Lock GetLock(System.Guid recordGuid, bool setLockConflict)
		{
			DataTable dtUser = DAL.LockManager.GetLock(recordGuid);

			if (dtUser.Rows.Count == 0)
				return(new BOL.Lock());
			else
				return(new BOL.Lock(dtUser.Rows[0]));       //Defect 251417 - moving database logic to DAL
		}

        /// <summary>
        /// Replacing the method() below..FormLockFunction is now local TODO
        /// </summary>
        /// <param name="recordGuid"></param>
        /// <param name="multiDivisionIndicator"></param>
        /// <returns></returns>
        public BOL.Lock LockRecord(System.Guid recordGuid, bool multiDivisionIndicator)
        {
            try
            {
                //CR3311 added lockinactivitytimeout
                System.Data.DataTable dtLock = DAL.LockManager.LockRecord(recordGuid, this.LockedFormId, this.LockFunction, multiDivisionIndicator, this.BaseLockedFormId, this._lockInactivityTimeoutMin);

                if (dtLock.Rows.Count == 0)
                    throw new BOL.BusinessObjectException("No locking information returned!");

                if (dtLock.Rows.Count > 1)
                    throw new BOL.BusinessObjectException("Locking returned too many rows?");

                BOL.Lock tmpLock = new BOL.Lock(dtLock.Rows[0]);

                this._lockingConflict = tmpLock.LockConflictIndicator(this.LockedFormId);

                //Add the lock to the collection if it was a valid lock
                if (!this._lockingConflict)
                {
                    this._isLockInitiator = true;
                    this.AddLock(tmpLock);
                }

                return (tmpLock);
            }
            catch (System.Data.SqlClient.SqlException err)
            {
                if (err.Number == 2627)		//Insert conflict, Record Locked for RecordGuid/Division, this 
                //should cover simultaneous lock conflicts
                {
                    BOL.Lock tmpLock = BOL.LockManager.GetLock(recordGuid, true);
                    return (tmpLock);
                }
                else
                    throw;
            }
        }


        /// <summary>
        /// Locks an entire Use Case (optional recordGuid)
        /// </summary>
        /// <param name="lockedRecordGuid">if Guid.Empty, its an entire UC lock</param>
        /// <param name="multiDivisionIndicator"></param>
        /// <returns></returns>
        public BOL.Lock LockUseCase(System.Guid lockedRecordGuid, bool multiDivisionIndicator)
        {
            //CR3311 added lockinactivitytimeout
            System.Data.DataTable dtLock = DAL.LockManager.LockUseCase(lockedRecordGuid, (int)this.LockFunction, this.LockedFormId, this.LockFunction, multiDivisionIndicator, this.BaseLockedFormId, this._lockInactivityTimeoutMin);

            if (dtLock.Rows.Count == 0)
                throw new BOL.BusinessObjectException("No locking information returned!");

            if (dtLock.Rows.Count > 1)
                throw new BOL.BusinessObjectException("Locking returned too many rows?");

            BOL.Lock tmpLock = new BOL.Lock(dtLock.Rows[0]);

            this._lockingConflict = tmpLock.LockConflictIndicator(this.LockedFormId);

            //Add the lock to the collection if it was a valid lock
            if (!_lockingConflict)
            {
                this._isLockInitiator = true;
                this.AddLock(tmpLock);
            }

            return (tmpLock);
        }

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/3/2005</CreationDate>
		/// <summary>
		/// Attempts to lock the supplied record if it's not: TODO: strip out LOCKFUNCTION to use local peropty
		/// </summary>
		/// <param name="recordGuid"></param>
		/// <param name="lockFunction"></param>
		///<param name="multiDivisionIndicator">Does the record lock span multi divisions (MUC03)</param>
		/// <returns></returns>
		public BOL.Lock LockRecord(System.Guid recordGuid, Common.LockFunctions lockFunction, bool multiDivisionIndicator)
		{
			try
			{
                //CR3311 added lockinactivitytimeout
                System.Data.DataTable dtLock = DAL.LockManager.LockRecord(recordGuid, this.LockedFormId, lockFunction, multiDivisionIndicator, this.BaseLockedFormId, this._lockInactivityTimeoutMin);

				if (dtLock.Rows.Count == 0)
					throw new BOL.BusinessObjectException("No locking information returned!");

				if (dtLock.Rows.Count > 1)
					throw new BOL.BusinessObjectException("Locking returned too many rows?");

				BOL.Lock tmpLock = new BOL.Lock(dtLock.Rows[0]);

			    this._lockingConflict = tmpLock.LockConflictIndicator(this.LockedFormId);

				//Add the lock to the collection if it was a valid lock
                if (!this._lockingConflict)
                {
                    this._isLockInitiator = true;
                    this.AddLock(tmpLock);
                }

                return(tmpLock);
			}
			catch(System.Data.SqlClient.SqlException err)
			{
				if (err.Number == 2627)		//Insert conflict, Record Locked for RecordGuid/Division, this 
											//should cover simultaneous lock conflicts
				{
					BOL.Lock tmpLock = BOL.LockManager.GetLock(recordGuid,true);
					return(tmpLock);
				}
				else
					throw;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/3/2005</CreationDate>
		/// <summary>
		/// Attempts to unlock the supplied records  -- TODO KILL ME after upgrade?
		/// </summary>
		/// <param name="recordGuids"></param>
        /// <param name="LockedFormId"></param>
		/// <param name="lockFunction"></param>
		public void UnlockRecords(System.Guid[] recordGuids, int LockedFormId, Common.LockFunctions lockFunction)
		{
			//Remove them from the database first...
			DAL.LockManager.UnlockRecords(recordGuids, LockedFormId, lockFunction );

			//Now, remove the lock from the collection
			foreach(System.Guid recordGuid in recordGuids)
			{
				this.RemoveLock(recordGuid, LockedFormId, lockFunction);

                //LOCKING420
                if (this.RecordGuids.Contains(recordGuid))
                {
                    this.RecordGuids.Remove(recordGuid);
                }
			}
		}

        /// <summary>
        /// LOCKING420 - Created after properties were moved from BaseForm to here
        /// </summary>
        public void UnlockRecords()
        {
            //Remove them from the database first...
            DAL.LockManager.UnlockRecords(this.RecordGuids, this.LockedFormId, this.LockFunction);

            //Now, remove the lock from the collection
            foreach (System.Guid recordGuid in this.RecordGuids)
            {
                this.RemoveLock(recordGuid, this.LockedFormId, this.LockFunction);
            }

            //Remove all the guids
            this.RecordGuids.Clear();
        }

		/// <summary>
		/// Adds a BOL.Lock object to our Lock connection - DB trip saving tool
		/// </summary>
		/// <param name="addingLock"></param>
		private void AddLock(BOL.Lock addingLock)
		{
			BOL.Lock tmpLock;

			// Justin Case
			if (this.IsRecordAlreadyLockedInUC(addingLock.LockedRecordGuid, out tmpLock))
				return;

			this._locks.Add(addingLock);

            //Keep these 2 in sync TODO: make this better
            if (!this.RecordGuids.Contains(addingLock.LockedRecordGuid))
                this.RecordGuids.Add(addingLock.LockedRecordGuid);
		}


		///<Developers>
		///	<Developer>Jason Engstrom</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/12/2005</CreationDate>
		/// <summary>
		/// Locked records Count
		/// </summary>
		public int LockCount
		{
			get
			{
				if (this._locks == null) 
					return(0);
				else
					return(this._locks.Count);
			}
		}

        /// <summary>
        /// LOCKING420 - Trying to help the developers out by not needing to create their own local arrays
        /// </summary>
        public System.Collections.ArrayList RecordGuids
        {
            get
            {
                return (this._recordGuidArray);
            }
        }

        /// <summary>
        /// Returns a user-friendly description of the LockFunction -- system reference table 
        /// </summary>
        /// <param name="lockFunctionId"></param>
        /// <returns></returns>
        public static string GUIFriendlyLockLocation(Common.LockFunctions lockFunctionId)
        {
            System.Data.DataRow drLock = BOL.ReferenceData.GetLockFunction(lockFunctionId);

            return(drLock[Common.VbecsTables.VbecsLockFunction.FunctionDescription].ToString());
        }

        /// <summary>
        /// Moved from VbecsBaseForm for code containment.
        /// Unlike LastUpdateFunctionId which corresponds to Forms, this corresponds to UC#
        /// </summary>
        public LockFunctions LockFunction
        {
            get
            {
                return (this._lockFunction);
            }
            set
            {
                this._lockFunction = value;
            }
        }

        /// <summary>
        /// A way for the GUI to check if there was any locking conflict (they can't get to the BOL.Lock collection)
        /// </summary>
        public bool LockingConflict
        {
            get
            {
                return (this._lockingConflict);
            }
        }

        /// <summary>
        /// ID of the form the lock belongs to. This is also stored on VBECSBaseForm, but I'm keeping a copy here to avoid 
        /// passing it in on all the non-static method calls, since LockManager cannot be instantiated without it
        /// </summary>
        public int LockedFormId
        {
            get
            {
                return (this._lockedFormId);
            }
        }

        /// <summary>
        /// Root form creating the locking chain
        /// </summary>
        public int BaseLockedFormId
        {
            get
            {
                return (this._baseLockedFormId);
            }
            set
            {
                this._baseLockedFormId = value;
            }
        }

		/// <summary>
		/// Removes a BOL.Lock object from our Lock connection
		/// </summary>
        /// <param name="lockedrecordGuid"></param>
        /// <param name="lockedFormId"></param>
		/// <param name="lockFunctionId"></param>
		private void RemoveLock(System.Guid lockedrecordGuid, int lockedFormId, Common.LockFunctions lockFunctionId)
		{
			foreach(BOL.Lock insertedLocks in this._locks)
			{
				if (insertedLocks.LockedRecordGuid == lockedrecordGuid &&
					insertedLocks.LockFunctionId == lockFunctionId &&
                    insertedLocks.LockedFormId == lockedFormId)
				{
					this._locks.Remove(insertedLocks);
					break;
				}
			}
		}		

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/3/2005</CreationDate>
		/// <summary>
		/// Unlocks all records for a Use-Case, FormInstance
		/// </summary>
		/// <param name="lockFunction"></param>
        /// <param name="LockedFormId"></param>
        /// <param name="BaseLockedFormId">baseform property</param>
		public void UnlockAllRecordsForUseCaseUser(Common.LockFunctions lockFunction, int LockedFormId, int BaseLockedFormId)
		{
            DAL.LockManager.UnlockAllRecordsForUseCaseUser(Common.LogonUser.LogonUserName, lockFunction, LockedFormId, BaseLockedFormId);
			this._locks.Clear();
            this.RecordGuids.Clear();
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/6/2004</CreationDate>
		/// <summary>
		/// Removes all locks for a particular user
		/// </summary>
		/// <param name="userName"></param>
		/// <param name="unlockAllSessions"></param>
		public static void UnlockAllRecordsForUser(string userName, bool unlockAllSessions)
		{
			DAL.LockManager.UnlockAllRecordsForUser(userName, unlockAllSessions);
		}
	}
}